using ConnectorLib.JSON;
using Pug.UnityExtensions;
using PugMod;
using PugTilemap;
using Unity.Entities;
using Unity.Mathematics;
using UnityEngine;
using Random = UnityEngine.Random;

namespace CrowdControl.Delegates.Effects.Implementations
{
    [Effect("teleport_core", "teleport_random")]
    public class Teleport : Effect
    {
        public Teleport(CrowdControl mod, NetworkClient client) : base(mod, client) { }

        public override EffectResponse Start(EffectRequest request)
        {
            PlayerController player = Manager.main.player;
            if (player == null) return EffectResponse.Failure(request.ID, StandardErrors.PlayerNotFound);

            World world = player.world;
            if (world is not { IsCreated: true })
                return EffectResponse.Failure(request.ID, StandardErrors.PlayerNotFound, "Player world component not ready.");

            string[] codeParams = request.code.Split('_');
            if (codeParams.Length < 2)
                return EffectResponse.Failure(request.ID, StandardErrors.BadRequest, "Invalid effect code format.");

            SinglePugMap pugMap = Object.FindObjectOfType<SinglePugMap>();
            if (pugMap == null)
                return EffectResponse.Failure(request.ID, StandardErrors.PlayerNotFound, "SinglePugMap not found in scene.");
            
            SinglePugMap.TileLayerLookup tileLayer = pugMap.GetTileLayerLookup();

            TileType lastTileType = 0;
            switch (codeParams[1])
            {
                case "core":
                {
                    player.QueueInputAction(new()
                    {
                        action = UIInputAction.Teleport,
                        position = float2.zero
                    });
                    return EffectResponse.Success(request.ID);
                }
                case "random":
                {
                    const int TELEPORT_RANGE = 1350;
                    int2 playerPos = player.WorldPosition.RoundToInt2();
                    Debug.Log($"Player position: {playerPos.x}, {playerPos.y}");
                    
                    for (int attempt = 0; attempt < 1000; attempt++)
                    {
                        int x = Random.Range(-TELEPORT_RANGE, TELEPORT_RANGE);
                        int z = Random.Range(-TELEPORT_RANGE, TELEPORT_RANGE);

                        int2 newPosition = playerPos + new int2(x, z); 
                        Debug.Log($"Attempt {attempt}: Trying position {newPosition.x}, {newPosition.y}");
                        
                        TileInfo tile = tileLayer.GetTopTile(newPosition);
                        Debug.Log($"Tile at position: {tile.tileType.ToString("G")}");
                        lastTileType = tile.tileType;
                        if (!tile.tileType.IsWalkableTile()) continue;

                        player.QueueInputAction(new()
                        {
                            action = UIInputAction.Teleport,
                            position = newPosition
                        });
                        return EffectResponse.Success(request.ID);
                    }
                    
                    return EffectResponse.Failure(request.ID, StandardErrors.NoValidTargets, "Could not find a walkable random position. Last tile type: " + lastTileType.ToString("G"));
                }
                default:
                    return EffectResponse.Failure(request.ID, StandardErrors.EffectUnknown);
            }
        }
    }
}